(function ($, window, document, joinchat_obj) {
  'use strict';

  joinchat_obj = $.extend({
    $div: null,
    settings: null,
    store: null,
    chatbox: false,
    showed_at: 0,
    is_ready: false, // Change to true when Joinchat ends initialization
    is_mobile: !!navigator.userAgent.match(/Android|iPhone|BlackBerry|IEMobile|Opera Mini/i),
    can_qr: window.QrCreator && typeof QrCreator.render == 'function',
  }, joinchat_obj);
  window.joinchat_obj = joinchat_obj; // Save global

  joinchat_obj.$ = function (sel) {
    return $(sel || this.$div, this.$div);
  };

  /**
   * Trigger Analytics events
   *
   * Available customizations via joinchat_obj.settings:
   *  - 'data_layer'    for custom data layer name (default 'dataLayer' or GTM4WP custom DataLayer name)
   *  - 'ga_event'      for GA4 custom event       (default 'generate_lead' recommended event)
   *
   * All params can be edited with document event 'joinchat:event' or cancel if returns false.
   * e.g.: $(document).on('joinchat:event', function(){ return false; });
   *
   */
  joinchat_obj.send_event = function (params) {
    params = $.extend({
      event_category: 'JoinChat', // Name
      event_label: '',            // Destination url
      event_action: '',           // "chanel: id"
      chat_channel: 'whatsapp',   // Channel name
      chat_id: '--',              // Channel contact (phone, username...)
      is_mobile: this.is_mobile ? 'yes' : 'no',
      page_location: location.href,
      page_title: document.title || 'no title',
    }, params);
    params.event_label = params.event_label || params.link || '';
    params.event_action = params.event_action || params.chat_channel + ': ' + params.chat_id;
    delete params.link;

    // Trigger event (params can be edited by third party scripts or cancel if return false)
    if (false === $(document).triggerHandler('joinchat:event', [params])) return;

    var data_layer = window[this.settings.data_layer] || window[window.gtm4wp_datalayer_name] || window['dataLayer'];
    var has_gtm = false;
    var has_gads = false;

    // GA4 send recomended event "generate_lead"
    var ga4_event = this.settings.ga_event || 'generate_lead';
    var ga4_params = $.extend({ transport_type: 'beacon' }, params);
    // Params already collected by GA4 (https://support.google.com/analytics/answer/9234069)
    delete ga4_params.page_location;
    delete ga4_params.page_title;
    // GA4 params max_length (https://support.google.com/analytics/answer/9267744)
    $.each(ga4_params, function (k, v) { if (typeof v == 'string') ga4_params[k] = v.substring(0, 100); });

    if (this.settings.gtag) {
      // gtag.js (New "Google Tag" find destinations)
      if (window.google_tag_data && google_tag_data.tidr && !!google_tag_data.tidr.destination) {
        for (var tag in google_tag_data.tidr.destination) {
          if (tag.substring(0, 2) == 'G-' || tag.substring(0, 3) == 'GT-') gtag('event', ga4_event, $.extend({ send_to: tag }, ga4_params)); // Send GA4 event
          else if (tag.substring(0, 4) == 'GTM-') has_gtm = true;
          else if (tag.substring(0, 3) == 'AW-') has_gads = true;
        }
      }
    } else {
      // gtag.js (Old method, traverse dataLayer and find 'config')
      if (typeof gtag == 'function' && typeof data_layer == 'object') {
        data_layer.forEach(function (item) {
          if (item[0] == 'config' && item[1] && (item[1].substring(0, 2) == 'G-' || item[1].substring(0, 3) == 'GT-'))
            gtag('event', ga4_event, $.extend({ send_to: item[1] }, ga4_params));
        });
        has_gtm = true;
        has_gads = true;
      }
    }

    // Send Google Ads conversion
    if (has_gads && this.settings.gads) {
      gtag('event', 'conversion', { send_to: this.settings.gads });
    }

    // Store category and delete from params
    var event_category = params.event_category;
    delete params.event_category;

    // Send Google Tag Manager custom event
    if (has_gtm && typeof data_layer == 'object') {
      data_layer.push($.extend({ event: event_category }, params));
    }

    // Send Facebook Pixel custom event
    if (typeof fbq == 'function') {
      fbq('trackCustom', event_category, params);
    }
  };

  // Return WhatsApp link with optional message
  joinchat_obj.whatsapp_link = function (phone, message, wa_web) {
    message = message !== undefined ? message : this.settings.message_send || '';
    wa_web = wa_web !== undefined ? wa_web : this.settings.whatsapp_web && !this.is_mobile;
    var link = (wa_web ? 'https://web.whatsapp.com/send?phone=' : 'https://wa.me/') + encodeURIComponent(phone || this.settings.telephone);

    return link + (message ? (wa_web ? '&text=' : '?text=') + encodeURIComponent(message) : '');
  };

  // Show Joinchat button
  joinchat_obj.show = function (tooltip) {
    this.$div.addClass('joinchat--show' + (!!tooltip ? ' joinchat--tooltip' : ''));
  };

  // Hide Joinchat button
  joinchat_obj.hide = function () {
    this.$div.removeClass('joinchat--show');
  };

  // Open Chatbox and trigger event
  joinchat_obj.chatbox_show = function () {
    if (!this.chatbox) {
      this.chatbox = true;
      this.showed_at = Date.now();
      this.$div.addClass('joinchat--chatbox');

      if (this.settings.message_badge && this.$('.joinchat__badge').hasClass('joinchat__badge--in')) {
        this.$('.joinchat__badge').toggleClass('joinchat__badge--in joinchat__badge--out');
      }
      // Trigger custom event
      $(document).trigger('joinchat:show');
    }
  };

  // Close Chatbox and trigger event
  joinchat_obj.chatbox_hide = function () {
    if (this.chatbox) {
      this.chatbox = false;
      this.$div.removeClass('joinchat--chatbox joinchat--tooltip');

      if (this.settings.message_badge) {
        this.$('.joinchat__badge').removeClass('joinchat__badge--out');
      }
      // Trigger custom event
      $(document).trigger('joinchat:hide');
    }
  };

  // Save CTA hash
  joinchat_obj.save_hash = function (force) {
    if (!this.settings.message_hash) return; // No hash
    if (this.settings.message_delay < 0 && !force) return; // No delay & no forced
    var saved_hashes = (this.store.getItem('joinchat_hashes') || '').split(',').filter(Boolean);

    if (saved_hashes.indexOf(this.settings.message_hash) === -1) {
      saved_hashes.push(this.settings.message_hash);
      this.store.setItem('joinchat_hashes', saved_hashes.join(','));
    }
  };

  // Open WhatsApp link with supplied phone and message or with settings defaults
  joinchat_obj.open_whatsapp = function (phone, message) {
    phone = phone || this.settings.telephone;
    message = message !== undefined ? message : this.settings.message_send || '';

    var params = {
      link: this.whatsapp_link(phone, message),
      chat_channel: 'whatsapp',
      chat_id: phone,
      chat_message: message,
    };
    var secure_link = new RegExp("^https?:\/\/(wa\.me|(api|web|chat)\.whatsapp\.com|" + location.hostname.replace('.', '\.') + ")\/.*", 'i');

    // Trigger event (params can be edited by third party scripts or cancel if return false)
    if (false === $(document).triggerHandler('joinchat:open', [params])) return;

    // Ensure the link is safe
    if (secure_link.test(params.link)) {
      // Send analytics events
      this.send_event(params);
      // Open WhatsApp link
      window.open(params.link, 'joinchat', 'noopener');
    } else {
      console.error("Joinchat: the link doesn't seem safe, it must point to the current domain or whatsapp.com");
    }
  };

  // Opt-in accepted or no needed
  joinchat_obj.optin = function () {
    return !this.$div.hasClass('joinchat--optout');
  };

  joinchat_obj.use_qr = function () {
    return !!this.settings.qr && this.can_qr && !this.is_mobile;
  }

  // Generate QR canvas
  joinchat_obj.qr = function (text, options) {
    var canvas = document.createElement('CANVAS');
    QrCreator.render($.extend({
      text: text,
      radius: 0.4,
      background: '#FFF',
      size: 200,
    }, this.settings.qr || {}, options || {}), canvas);
    return canvas;
  }

  function joinchat_magic() {
    $(document).trigger('joinchat:starting');

    var button_delay = joinchat_obj.settings.button_delay * 1000;
    var chat_delay = Math.max(0, joinchat_obj.settings.message_delay * 1000);
    var has_cta = !!joinchat_obj.settings.message_hash;
    var has_chatbox = !!joinchat_obj.$('.joinchat__box').length;
    var timeoutHover, timeoutCTA;

    // Stored values
    var has_pageviews = parseInt(joinchat_obj.store.getItem('joinchat_views') || 1) >= joinchat_obj.settings.message_views;
    var saved_hashes = (joinchat_obj.store.getItem('joinchat_hashes') || '').split(',').filter(Boolean);
    var is_viewed = joinchat_obj.settings.is_viewed !== undefined ? joinchat_obj.settings.is_viewed :
      saved_hashes.indexOf(joinchat_obj.settings.message_hash || 'none') !== -1;

    function clear_and_show() {
      clearTimeout(timeoutCTA);
      joinchat_obj.chatbox_show();
    }

    function save_and_hide() {
      joinchat_obj.save_hash();
      joinchat_obj.chatbox_hide();
    }

    function joinchat_click() {
      if (has_chatbox && !joinchat_obj.chatbox) {
        clear_and_show();
      } else if (Date.now() > joinchat_obj.showed_at + 600) { // A bit delay to prevent open WA on auto show
        save_and_hide();
        joinchat_obj.open_whatsapp();
      }
    }

    // Show button (and tooltip)
    var has_tooltip = !is_viewed && (joinchat_obj.settings.message_badge || !has_cta || !chat_delay || !has_pageviews);
    setTimeout(function () { joinchat_obj.show(has_tooltip); }, button_delay);

    // Show badge or chatbox
    if (has_cta && chat_delay && !is_viewed) {
      if (joinchat_obj.settings.message_badge) {
        timeoutCTA = setTimeout(function () { joinchat_obj.$('.joinchat__badge').addClass('joinchat__badge--in'); }, button_delay + chat_delay);
      } else if (has_pageviews) {
        timeoutCTA = setTimeout(clear_and_show, button_delay + chat_delay);
      }
    }

    // Open Chatbox on mouse over
    if (has_chatbox && !joinchat_obj.is_mobile) {
      joinchat_obj.$('.joinchat__button')
        .on('mouseenter', function () { timeoutHover = setTimeout(clear_and_show, 1500); })
        .on('mouseleave', function () { clearTimeout(timeoutHover); });
    }

    joinchat_obj.$('.joinchat__button').on('click', joinchat_click);
    joinchat_obj.$('.joinchat__close').on('click', save_and_hide);

    // Opt-in toggle
    joinchat_obj.$('#joinchat_optin').on('change', function () {
      joinchat_obj.$div.toggleClass('joinchat--optout', !this.checked);
    });

    // Only scroll Joinchat message box (no all body)
    // TODO: disable also on touch
    joinchat_obj.$('.joinchat__box__scroll').on('mousewheel DOMMouseScroll', function (e) {
      e.preventDefault();
      var delta = e.originalEvent.wheelDelta || -e.originalEvent.detail;
      this.scrollTop += (delta < 0 ? 1 : -1) * 30;
    });

    // Mobile enhancements
    if (joinchat_obj.is_mobile) {
      var timeoutKB, timeoutResize;

      function form_focus_toggle() {
        var type = (document.activeElement.type || '').toLowerCase();

        if (['date', 'datetime', 'email', 'month', 'number', 'password', 'search', 'tel', 'text', 'textarea', 'time', 'url', 'week'].indexOf(type) >= 0) {
          if (joinchat_obj.chatbox) {
            joinchat_obj.chatbox_hide();
            setTimeout(function () { joinchat_obj.hide(); }, 400);
          } else {
            joinchat_obj.hide();
          }
        } else {
          joinchat_obj.show();
        }
      }

      // Hide on mobile when virtual keyboard is open (on fill forms)
      $(document).on('focus blur', 'input, textarea', function (e) {
        if (!$(e.target).closest(joinchat_obj.$div).length) {
          clearTimeout(timeoutKB);
          timeoutKB = setTimeout(form_focus_toggle, 200);
        }
      });

      // Ensure header is visible
      $(window).on('resize', function () {
        clearTimeout(timeoutResize);
        timeoutResize = setTimeout(function () { joinchat_obj.$div[0].style.setProperty('--vh', window.innerHeight + 'px'); }, 200);
      }).trigger('resize');
    }

    // Triggers: open chatbox on load if query or anchor "joinchat" exists
    var location_url = new URL(window.location);
    if (location_url.hash == '#joinchat') {
      joinchat_obj.show();
      setTimeout(clear_and_show, 700); // 500ms animation + 200ms extra delay
    }
    if (location_url.searchParams.has('joinchat')) {
      var query_delay = (parseInt(location_url.searchParams.get('joinchat')) || 0) * 1000;
      setTimeout(function () { joinchat_obj.show(); }, query_delay);
      setTimeout(clear_and_show, query_delay + 700);
    }

    // Triggers: open chatbox or launch WhatsApp on click
    $(document).on('click', '.joinchat_open, .joinchat_app, a[href="#joinchat"], a[href="#whatsapp"]', function (e) {
      e.preventDefault();
      if (has_chatbox && (!joinchat_obj.optin() || $(this).is('.joinchat_open, a[href="#joinchat"]'))) clear_and_show(); // Open chatbox
      else joinchat_obj.open_whatsapp($(this).data('phone'), $(this).data('message')); // WhatsApp direct
    });

    // Triggers: close chatbox when click on nodes with class "joinchat_close"
    $(document).on('click', '.joinchat_close', function (e) {
      e.preventDefault();
      joinchat_obj.chatbox_hide();
    });

    // Triggers: open chatbox on scroll (when node on viewport)
    var $show_on_scroll = $('.joinchat_show, .joinchat_force_show');
    if (has_cta && $show_on_scroll.length && 'IntersectionObserver' in window) {
      var observer = new IntersectionObserver(function (objs) {
        $.each(objs, function () {
          if (this.intersectionRatio <= 0) return;
          var is_forced = this.target.classList.contains('joinchat_force_show');
          if (is_viewed && !is_forced) return;

          observer.disconnect(); // Only one show for visit
          joinchat_obj.save_hash(!is_forced);
          clear_and_show();
          return false;
        });
      });
      $show_on_scroll.each(function () { observer.observe(this); });
    }

    // Add QR Code
    if (joinchat_obj.use_qr()) {
      joinchat_obj.$('.joinchat__qr').append(joinchat_obj.qr(joinchat_obj.whatsapp_link(undefined, undefined, false)));
    } else {
      joinchat_obj.$('.joinchat__qr').remove();
    }

    // Fix message clip-path style broken by some CSS optimizers
    if (has_chatbox) {
      joinchat_obj.$div.css('--peak', 'ur' + 'l(#joinchat__peak_' + (joinchat_obj.$div.closest('[dir=rtl]').length ? 'r' : 'l') + ')');
    }

    // Count visits (if needed)
    if (chat_delay && !has_pageviews) {
      joinchat_obj.store.setItem('joinchat_views', parseInt(joinchat_obj.store.getItem('joinchat_views') || 0) + 1);
    }

    $(document).trigger('joinchat:start');
    joinchat_obj.is_ready = true;
  }

  // Simple run only once wrapper
  function once(fn) {
    return function () {
      fn && fn.apply(this, arguments);
      fn = null;
    };
  }

  function on_page_ready() {
    joinchat_obj.$div = $('.joinchat');

    // Exit if no joinchat div
    if (!joinchat_obj.$div.length) return;

    joinchat_obj.settings = joinchat_obj.$div.data('settings');

    // Fallback if localStorage not supported (iOS incognito)
    // Implements functional storage in memory and will not persist between page loads
    try {
      localStorage.setItem('test', 1);
      localStorage.removeItem('test');
      joinchat_obj.store = localStorage;
    } catch (e) {
      joinchat_obj.store = {
        _data: {},
        setItem: function (id, val) { this._data[id] = String(val); },
        getItem: function (id) { return this._data.hasOwnProperty(id) ? this._data[id] : null; }
      };
    }

    // In some strange cases data settings are empty
    if (typeof joinchat_obj.settings != 'object') {
      try {
        joinchat_obj.settings = JSON.parse(joinchat_obj.$div.attr('data-settings'));
      } catch (error) {
        joinchat_obj.settings = undefined;
        console.error("Joinchat: can't get settings");
      }
    }

    // Only works if joinchat is defined
    if (!!joinchat_obj.settings && !!joinchat_obj.settings.telephone) {
      if (joinchat_obj.is_mobile || !joinchat_obj.settings.mobile_only) {
        joinchat_magic();
      } else {
        // Ensure don't show
        joinchat_obj.hide();

        // Triggers: launch WhatsApp on click
        $(document).on('click', '.joinchat_open, .joinchat_app, a[href="#joinchat"], a[href="#whatsapp"]', function (e) {
          e.preventDefault();
          joinchat_obj.open_whatsapp($(this).data('phone'), $(this).data('message'));
        });
      }

      // Gutenberg buttons add QR
      if (joinchat_obj.can_qr && !joinchat_obj.is_mobile) {
        $('.joinchat-button__qr').each(function () {
          $(this).append(joinchat_obj.qr(joinchat_obj.whatsapp_link($(this).data('phone'), $(this).data('message'), false)));
        });
      } else {
        $('.wp-block-joinchat-button figure').remove();
      }

      // Replace product variable SKU
      if (joinchat_obj.settings.sku !== undefined) {
        var message = joinchat_obj.settings.message_send;
        $('form.variations_form').on('found_variation reset_data', function (e, variation) {
          var sku = variation && variation.sku || joinchat_obj.settings.sku;
          joinchat_obj.$('.joinchat__box sku').text(sku);
          joinchat_obj.settings.message_send = message.replace(/<sku>.*<\/sku>/g, sku);
        });
      }
    }
  }

  // Ready!! (in some scenarios jQuery.ready doesn't fire, this try to ensure Joinchat initialization)
  var once_page_ready = once(on_page_ready);
  $(once_page_ready);
  $(window).on('load', once_page_ready);
  document.addEventListener('DOMContentLoaded', once_page_ready);

}(jQuery, window, document, window.joinchat_obj || {}));
